import { Suspense } from "react"
import { notFound } from "next/navigation"
import { Header } from "@/components/header"
import { PersonDetails } from "@/components/person-details"
import { PersonCredits } from "@/components/person-credits"
import { PersonImages } from "@/components/person-images"
import { Skeleton } from "@/components/ui/skeleton"

async function getPerson(id: string) {
  try {
    const res = await fetch(
      `https://api.themoviedb.org/3/person/${id}?api_key=${process.env.TMDB_API_KEY}&append_to_response=movie_credits,tv_credits,images`,
      { next: { revalidate: 3600 } },
    )

    if (!res.ok) {
      console.error(`TMDB person fetch failed (${res.status}) for id ${id}`)
      return null
    }

    return res.json()
  } catch (err) {
    console.error("TMDB person fetch error:", err)
    return null
  }
}

export default async function PersonPage({
  params,
}: {
  params: { id: string }
}) {
  const person = await getPerson(params.id)

  if (!person) {
    notFound()
  }

  return (
    <div className="min-h-screen bg-background">
      <Header />

      <div className="container mx-auto px-4 py-8">
        <PersonDetails person={person} />

        <div className="mt-12 space-y-12">
          <div>
            <h2 className="text-2xl font-bold mb-6">Filmography</h2>
            <Suspense fallback={<PersonCreditsSkeleton />}>
              <PersonCredits movieCredits={person.movie_credits} tvCredits={person.tv_credits} />
            </Suspense>
          </div>

          {person.images?.profiles && person.images.profiles.length > 0 && (
            <div>
              <h2 className="text-2xl font-bold mb-6">Photos</h2>
              <Suspense fallback={<PersonImagesSkeleton />}>
                <PersonImages images={person.images.profiles} />
              </Suspense>
            </div>
          )}
        </div>
      </div>
    </div>
  )
}

function PersonCreditsSkeleton() {
  return (
    <div className="space-y-6">
      <div className="grid grid-cols-2 md:grid-cols-4 lg:grid-cols-6 gap-4">
        {Array.from({ length: 12 }).map((_, i) => (
          <div key={i} className="space-y-2">
            <Skeleton className="aspect-[2/3] w-full rounded-lg" />
            <Skeleton className="h-4 w-full" />
            <Skeleton className="h-3 w-2/3" />
          </div>
        ))}
      </div>
    </div>
  )
}

function PersonImagesSkeleton() {
  return (
    <div className="grid grid-cols-2 md:grid-cols-4 lg:grid-cols-6 gap-4">
      {Array.from({ length: 12 }).map((_, i) => (
        <Skeleton key={i} className="aspect-[2/3] w-full rounded-lg" />
      ))}
    </div>
  )
}

export async function generateMetadata({
  params,
}: {
  params: { id: string }
}) {
  const person = await getPerson(params.id)

  if (!person) {
    return {
      title: "Person Not Found",
    }
  }

  return {
    title: `${person.name} - MoviesWeb`,
    description: person.biography || `Learn more about ${person.name}`,
    openGraph: {
      title: person.name,
      description: person.biography || `Learn more about ${person.name}`,
      images: person.profile_path ? [`https://image.tmdb.org/t/p/w500${person.profile_path}`] : [],
    },
  }
}
